<?php
###############################################################################
# Adding SEMRush data into domain database.
#
# Rename thirdparty/config-sample.php to config.php
# Edit it to include the ID and KEY from SEMRush
# Run this as a cron-job at regular intervals to insert data into the
# domain table.
# eg.
#
# */3 * * * * php /path/to/wmdsed3/thirdparty/semrush.php d=90 >/dev/null 2>&1
# or
# */5 * * * * wget -q -O /dev/null "http://<install-url>/thirdparty/semrush.php?d=90" >/dev/null 2>&1
#
###############################################################################

require_once(dirname(__FILE__) . "/../include.php");
require_once(dirname(__FILE__) . "/config.check.php");
require_once(dirname(__FILE__) . "/../lib/php/idn/idn.php");

###############################################################################
# Parse command line if running in CLI

if(is_cli())
	parse_str(implode('&', array_slice($argv, 1)), $_REQUEST);

###############################################################################

loginIfRequired();

###############################################################################

if(!function_exists('curl_version'))
{
	smartEcho("Please install php5-curl.");
	exit;
}

###############################################################################
# Get Command-line

$verbose        = false;
$refresh_days   = 30;
$semrush_batch_size = 1;

if(isset($_REQUEST['v']))
	$verbose = true;
if(isset($_REQUEST['d']))
{
	$days = intval($_REQUEST['d']);
	if($days >= -1 && $days <= 365)
		$refresh_days = $days;
}


###############################################################################
# Open the Domain Database

$pdo = init_db();
if($pdo === false)
{
	smartEcho("Unable to connect to database");
	exit;
}

###############################################################################
# Add the MOZ columns to the domain portfolio database.
# NOTE: This may fail if any of the columns is already present in the 
# domain table but not registered as a custom column. If you have problems
# you will need to start with a fresh DB.

$columns = $pdo->getAllColumnNames();
if($columns !== false)
{
	# date time column
	if(!in_array('semrush_checked_at', $columns))
		$pdo->addDateTimeColumn('semrush_checked_at');

	# Number columns
	$required_columns = array('semrush_rank', 'semrush_kw', 'semrush_ot', 'semrush_oc');
	foreach($required_columns as $col)
	{
		if(!in_array($col, $columns))
		{
			$pdo->addBigIntColumn($col, 20);
			if($verbose)
				smartEcho("Added custom column: $col");
		}
	}
}
else
{
	smartEcho($pdo->getLastError());
	exit;
}

###############################################################################
# Find the domains to lookup.

$batchedDomains = array();

$lookup_array = $pdo->getDomainsToLookup("semrush_checked_at", $semrush_batch_size, $refresh_days);
if($lookup_array !== false)
{
	foreach($lookup_array as $dinfo)
	{
		$domain = $dinfo['domain'];
		$domain = idn_convert($domain);
		$batchedDomains[] = $domain;
		if($verbose)
			smartEcho("Found $domain");
	}
	if($verbose)
	{
		if(!count($batchedDomains))
		{
			smartEcho("Nothing to do!");
			exit;
		}
	}
}
else
{
	smartEcho($pdo->getLastError());
	exit;
}

###############################################################################
# Find the SEMRush Data

foreach($batchedDomains as $domain)
{
	$data_array = get_semrush_data($domain, $semrush_api_key, $semrush_db_region);
	if($data_array !== false)
	{
		$data_array['semrush_checked_at'] = date("Y-m-d H:i:s");
		$pdo->setDomainDataFromArray($domain, $data_array);
		if($verbose)
			DebugPrintArray($data_array);
	}
	else
	{
		$data_array = array();
		$data_array['semrush_checked_at'] = date("Y-m-d H:i:s");
		$pdo->setDomainDataFromArray($domain, $data_array);
	}
}

###############################################################################

function api_http_call($url)
{
	$options = array(CURLOPT_RETURNTRANSFER => true);
	$ch = curl_init($url);
	curl_setopt_array($ch, $options);
	$content = curl_exec($ch);
	$http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
	curl_close( $ch );
	if($http_status == 200)
	{
		$parts = explode("\n", trim($content));
		return $parts;
	}
	return false;
}

###############################################################################
# The SEMRUSH API Call
/*
Domain;Rank;Organic Keywords;Organic Traffic;Organic Cost
apple.com;20;2457273;10350767;8731399
*/

function get_semrush_data($domain, $secret_key, $dbregion)
{
	$requestUrl = "http://api.semrush.com/?type=domain_rank&key={$secret_key}&export_columns=Dn,Rk,Or,Ot,Oc&domain={$domain}&database={$dbregion}";
	$srdata = api_http_call($requestUrl);

	if(is_array($srdata) && count($srdata) == 2)
	{
		$results = $srdata[1];
		$parts = explode(";", $results);
		if(count($parts) == 5)
		{
			$semdata = array();
			$semdata['semrush_rank'] = intval($parts[1]);
			$semdata['semrush_kw'] = intval($parts[2]);
			$semdata['semrush_ot'] = intval($parts[3]);
			$semdata['semrush_oc'] = intval($parts[4]);
			return $semdata;
		}
	}
	return false;
}

###############################################################################

?>
