<?php
###############################################################################
# Adding Estibot data into domain database.
#
# Rename thirdparty/config-sample.php to config.php
# Edit it to include the auth-data from Estibot.
#
# Run this as a cron-job at regular intervals to insert data into the
# domain table.
# eg.
#
# */3 * * * * php /path/to/wmdsed3/thirdparty/estibot.php d=90 >/dev/null 2>&1
# or
# */3 * * * * wget -q -O /dev/null "http://<install-url>/thirdparty/estibot.php?d=90" >/dev/null 2>&1
#
###############################################################################

header('Content-Type: text/html; charset=utf-8');

###############################################################################

require_once(dirname(__FILE__) . "/../include.php");
require_once(dirname(__FILE__) . "/config.check.php");
require_once(dirname(__FILE__) . "/../lib/php/idn/idn.php");

###############################################################################
# Parse command line if running in CLI

if(is_cli())
	parse_str(implode('&', array_slice($argv, 1)), $_REQUEST);

###############################################################################

loginIfRequired();

###############################################################################

if(!function_exists('curl_version'))
{
	smartEcho("Please install php5-curl.");
	exit;
}

###############################################################################

if(!extension_loaded('simplexml')) 
{
	smartEcho("Please install simplexml extension.");
	exit;
}

###############################################################################
# Get Command-line

$verbose        = false;
$refresh_days   = 30;
$domain_batch_size = 5;

if(isset($_REQUEST['v']))
	$verbose = true;
if(isset($_REQUEST['d']))
{
	$days = intval($_REQUEST['d']);
	if($days >= -1 && $days <= 365)
		$refresh_days = $days;
}
if(isset($_REQUEST['b']))
{
	$batch = intval($_REQUEST['b']);
	if($batch > 0 && $batch <= 100)
		$domain_batch_size = $batch;
}


###############################################################################
# Open the Domain Database

$pdo = init_db();
if($pdo === false)
{
	smartEcho("Unable to connect to database");
	exit;
}

###############################################################################
# Add the Estibot columns to the domain portfolio database.
# NOTE: This may fail if any of the columns is already present in the 
# domain table but not registered as a custom column. If you have problems
# you will need to start with a fresh DB.

$columns = $pdo->getAllColumnNames();
if($columns !== false)
{
	# date time column
	if(!in_array('eb_checked_at', $columns))
		$pdo->addDateTimeColumn('eb_checked_at');
	# Float Columns
	if(!in_array('eb_tm_probability', $columns))
		$pdo->addFloatColumn('eb_tm_probability', "6.2");
	if(!in_array('eb_exact_avg_cpc', $columns))
		$pdo->addFloatColumn('eb_exact_avg_cpc', "6.2");
		
	# Long Text Column (not user interface visible)
	if(!in_array('eb_xml_data', $columns))
	{
		$dbcolumns[] = "eb_xml_data";
		$dbcolumns[] = "longtext";
		$dbcolumns[] = "DEFAULT NULL";
		$pdo->AddColumnToMainTable($dbcolumns);
	}
	
	# Text columns
	$required_text_columns = array('eb_category', 'eb_language', 'eb_tm_risk', 'eb_exact_match_kw', 'eb_exact_avg_comp');
	foreach($required_text_columns as $tcol)
	{
		if(!in_array($tcol, $columns))
			$pdo->addTextColumn($tcol, "64");
	}
	
	# Number columns
	$required_num_columns = array('eb_appraised_value', 
	'eb_search_results_phrase', 'eb_search_ads', 'eb_exact_avg_search_volume');
	foreach($required_num_columns as $col)
	{
		if(!in_array($col, $columns))
			$pdo->addBigIntColumn($col, 20);
	}
}
else
{
	smartEcho($pdo->getLastError());
	exit;
}

###############################################################################
# Find the domains to lookup.

$domain_names = "";
$domain_array = array();

$lookup_array = $pdo->getDomainsToLookup("eb_checked_at", $domain_batch_size, $refresh_days);
if($lookup_array !== false)
{
	foreach($lookup_array as $dinfo)
	{
		$domain = $dinfo['domain'];
		$domain = idn_convert($domain);
		$domain_array[] = $domain;
		if($domain_names != "")
			$domain_names .= ">>";
		$domain_names .= $domain;
	}
	if($verbose)
		smartEcho($domain_names);
}
else
{
	smartEcho($pdo->getLastError());
	exit;
}

###############################################################################
	
$data_array = get_estibot_data($domain_names, $estibot_login, $estibot_password, $verbose);
if($data_array !== false)
{	
	foreach($data_array as $estidata)
	{
		$dataarray = array();
		$domain = "";
		foreach($estidata as $column=>$cdata)
		{
			if($column != 'eb_xml_data' && $verbose)
				smartEcho("$column : $cdata");
			if($column != 'eb_domain')
				$dataarray[$column] = $cdata;
			else 
			{
				$domain = strtolower($cdata);
				$domain = idn_reconvert($domain);
			}
		}
		if($domain != "")
		{
			$dataarray['eb_checked_at'] = date("Y-m-d H:i:s");
			$pdo->setDomainDataFromArray($domain, $dataarray);
		}
	}
}

if($verbose && !count($domain_array))
	smartEcho("Nothing to do");

###############################################################################

function get_estibot_data($domain_name, $login, $password, $verbose=false)
{
	$estibotValueURL = "http://www.estibot.com/api.php?a=get&email=$login&password=$password&c=appraise&t=$domain_name";
	
	$contents = estibot_api_http_call($estibotValueURL);
	$data_array = array();
	if($contents !== false)
	{
		# Remove the bad XML output by estibot
		$contents = replace_content_inside_delimiters("<keyword_exact_ntld_ng>", "</keyword_exact_ntld_ng>", "", $contents);
		# 
		$xml = new SimpleXMLElement($contents);
		foreach($xml->appraisal as $ap)
		{
			$estimates = array();
			$estimates['eb_xml_data'] = $ap->asXML();
			$estimates['eb_domain'] = (string)$ap->domain_cc;
			$estimates['eb_appraised_value'] = intval((string)$ap->appraised_value);
			$estimates['eb_category'] = (string)$ap->category;
			$estimates['eb_language'] = (string)$ap->language;
			//$estimates['eb_appraised_wholesale_value'] = intval($ap->appraised_wholesale_value);
			//$estimates['eb_appraised_monetization_value'] = intval($ap->appraised_monetization_value);
			$estimates['eb_search_results_phrase'] = intval($ap->search_results_phrase);
			$estimates['eb_search_ads'] = intval($ap->search_ads);
			$estimates['eb_tm_risk'] = (string)$ap->trademark_risk;
			$estimates['eb_tm_probability'] = (float)$ap->trademark_probability;
			
			if(isset($ap->keyword_stats->search))
			{
				$sd = $ap->keyword_stats->search;
				foreach($sd as $sde)
				{
					if((string)$sde->type == "exact")
					{	
						$sda = $sde->attributes();
						if(isset($sda))
						{
							foreach($sde->attributes() as $kw)
							{
								$estimates['eb_exact_match_kw'] = (string)$kw;
								break;
							}
						}
						$estimates['eb_exact_avg_comp'] = (string)$sde->avg_competition;
                        $estimates['eb_exact_avg_cpc'] = (float)$sde->avg_cpc;
                        $estimates['eb_exact_avg_search_volume'] = intval($sde->avg_search_volume);
                        //$estimates['eb_exact_local_search_volume'] = intval($sde->local_search_volume);
					}
				}
			}
			$data_array[] = $estimates;
		}
		return $data_array;
	}
	return false;
}

###############################################################################

function estibot_api_http_call($url)
{
	$options = array(CURLOPT_RETURNTRANSFER => true);
	$ch = curl_init($url);
	curl_setopt_array($ch, $options);
	$content = curl_exec($ch);
	$http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
	curl_close( $ch );
	if($http_status == 200)
		return $content; 
	return false;
}

###############################################################################

function replace_content_inside_delimiters($start, $end, $new, $source) 
{
	return preg_replace('#('.preg_quote($start).')(.*?)('.preg_quote($end).')#si', '$1'.$new.'$3', $source);
}

###############################################################################

?>
