<?php
###############################################################################
# lookup.php
# Watch My Domains SED v3
# (c) Softnik Technologies. All rights reserved.
###############################################################################

require_once(get_php_folder_path() . "idn/idn.php");
require_once("dns.php");
require_once("whois.php");
require_once("http.php");
require_once("googlepr.php");
require_once("alexa.php");
require_once("mx.php");
require_once("ssl.php");
require_once("googleindex.php");

###############################################################################
// Do Lookup
// Find the next lookup candidate entry in queue 
// Mark it's status as "in progress" and set last_updated time.
// Finish the lookup and set it's status as completed or failed. 
// Set last_updated time.

function processLookupQueue($max_count, $start_from=0, $prefer="")
{
    $pdo = init_db();
	if($pdo === false)
		return false;
        
	$lookup_array = $pdo->getFromLookupQueue($max_count, $start_from, $prefer);
    if($lookup_array === false)
	{
        return false;
	}

	# Prefered lookup queue is empty. So process the rest.
	if(count($lookup_array) == 0 && $prefer != "")
		return processLookupQueue($max_count, $start_from, "");
		
	$subdomain = getConfigData("config_subdomain_entry", "www");
	$processed = 0;
	foreach($lookup_array as $row)
	{
		$sid           = $row['sid'];
		$lookup_type   = $row['type'];
		$lookup_status = $row['status'];
		$server        = $row['server'];
		$domain_name   = $pdo->getSingleEntry($pdo->getDomainTableName(), "sid", $sid, "domain");
		$status        = true;
		if($lookup_status == LUQSTAT_WAITING)
		{
			switch($lookup_type)
			{
				case LOOKUP_DOMAIN_WHOIS:
					$status = doDomainRegistryWhois($domain_name);
					break;
				case LOOKUP_SEC_DOMAIN_WHOIS:
					$status = doDomainRegistrarWhois($domain_name, $server);
					break;
				case LOOKUP_IP_ADDRESS:
					do_ip_lookup($domain_name, $subdomain);
					break;
				case LOOKUP_HTTP_HOMEPAGE:
					get_and_parse_homepage($domain_name, $subdomain);
					break;
				case LOOKUP_ALEXA_DATA:
					get_alexa_records($domain_name, $subdomain);
					break;
				case LOOKUP_MX_RECORDS:
					get_mx_records($domain_name);
					break;
				case LOOKUP_PING_RESPONSE:
					get_ping_response($domain_name, $subdomain);
					break;
				case LOOKUP_YAHOO_LINKS:
					break;
				case LOOKUP_GOOGLE_LINKS:
					break;
				//case LOOKUP_GOOGLE_PR:
				//	get_google_pr($domain_name, $subdomain);
				//	break;
				case LOOKUP_SSL_CERTS:
					get_all_ssl_records($domain_name);
					break;
				case LOOKUP_IP_WHOIS:
					doWhoisForDomainIP($domain_name);
					break;
				case LOOKUP_SUB_DOMAINS:
					do_subdomain_lookup($domain_name);
					break;
				case LOOKUP_GOOGLE_INDEX:
					$status = get_google_index($domain_name, $subdomain);
					break;
			}
			if($status === false)
			{
				if($lookup_type == LOOKUP_SEC_DOMAIN_WHOIS)
					$pdo->addToRegistrarWhoisLookupQueue($sid, $server, 0);
				else if($lookup_type == LOOKUP_DOMAIN_WHOIS)
					$pdo->addToLookupQueue($sid, $lookup_type, 0);
				else if($lookup_type == LOOKUP_GOOGLE_INDEX)
					$pdo->addToLookupQueue($sid, $lookup_type, 0);
			}
		}
		if($status !== false)
			$processed++;
	}
	
	// Nothing got processed trying preferred lookups, 
	// let us achieve something.
	if($processed < 3 && $prefer != "" && count($lookup_array))
	{
		$maxc = ($max_count/2) - $processed;
		// Do we have the time for processing all?
		global $script_started_at;
		if(isset($script_started_at))
		{
			$set = microtime(true);
			$ext = round($set-$script_started_at);
			if($ext < 8)
				$maxc = $max_count-$processed;
		}
		if($maxc == 0)
			$maxc = 1;
		$extraproc = processLookupQueue($maxc, $start_from, "");
		if($extraproc !== false)
			$processed += $extraproc['processed'];
	}
	
	if(count($lookup_array))
		$pdo->logQueueLookupTimestamp();
	
	$pinfo = array('count'=>0, 'processed'=>$processed );
	$queuesize = $pdo->getCount($pdo->getLookupQueueTableName());
	if($queuesize !== false)
		$pinfo['count'] = $queuesize;
	return $pinfo;
}

###############################################################################

function getLookupTypeFromCode($lucode)
{
	$luwhat = 0;
	$luw = strtolower(trim($lucode));
	if($luw == "alexa")
		$luwhat = intval(LOOKUP_ALEXA_DATA);
	else if($luw == "dw")
		$luwhat = intval(LOOKUP_DOMAIN_WHOIS);
	else if($luw == "sdw")
		$luwhat = intval(LOOKUP_SEC_DOMAIN_WHOIS);
	//else if($luw == "gpr")
	//	$luwhat = intval(LOOKUP_GOOGLE_PR);
	else if($luw == "ip")
		$luwhat = intval(LOOKUP_IP_ADDRESS);
	else if($luw == "mx")
		$luwhat = intval(LOOKUP_MX_RECORDS);
	else if($luw == "ipw")
		$luwhat = intval(LOOKUP_IP_WHOIS);
	else if($luw == "http")
		$luwhat = intval(LOOKUP_HTTP_HOMEPAGE);
	else if($luw == "ping")
		$luwhat = intval(LOOKUP_PING_RESPONSE);
	else if($luw == "ssl")
		$luwhat = intval(LOOKUP_SSL_CERTS);
	else if($luw == "dns")
		$luwhat = intval(LOOKUP_SUB_DOMAINS);
	else if($luw == "gi")
		$luwhat = intval(LOOKUP_GOOGLE_INDEX);
	return $luwhat;
}

###############################################################################

function addDomainsToLookupQueue($catid, $maxdays, $lookuptypes, $maxcount, $ri=0)
{
	$added = 0;
	$pdo = init_db();
	if($pdo !== false)
	{
		$lutcodes = explode(",", $lookuptypes);
		foreach($lutcodes as $lucode)
		{
			$luwhat = getLookupTypeFromCode($lucode);
			$luadded = 0;
			if($luwhat != 0)
			{
				$last_lookup_time_field = $pdo->getLastLookupTimeField($luwhat);
				$mysqldate = $pdo->MySQLDateOffset(date("Y-m-d"),0,0,-$maxdays) . " 23:59:59";
				$domaintable = $pdo->getDomainTableName();
				$catconntable = $pdo->getCategoryIDToDomainIDTableName();
				$sql = "SELECT DISTINCT a.sid,a.domain FROM $domaintable a, $catconntable b WHERE (a.$last_lookup_time_field <= ? OR a.$last_lookup_time_field IS NULL OR a.$last_lookup_time_field = '0000-00-00') AND ((a.sid = b.sid AND b.CategoryID = ?) OR $catid=1)";
				if($maxcount > 1)
					$sql .= " LIMIT 0,$maxcount";
				try
				{
					$q = $pdo->db_connect_handle->prepare($sql);
					$q->execute(array($mysqldate,$catid));
					$rows = $q->fetchAll(PDO::FETCH_ASSOC);
					if($rows !== false && is_array($rows))
					{
						foreach($rows as $row)
						{
							$sid = $row['sid'];
							$domain = $row['domain'];
							if($pdo->addToLookupQueue($sid, $luwhat, $ri) !== false)
							{
								$luadded++;
							}
						}
					}
				}
				catch (PDOException $e) 
				{
					$pdo->setError($e->getMessage());
				}
			}
			$added += $luadded;
		}
	}
	return $added;
}

###############################################################################

function addRecentDomainsToWhoisLookupQueue($catid, $maxdays, $ri=0)
{
	$added = 0;
	$pdo = init_db();
	if($pdo !== false)
	{
		$luwhat = LOOKUP_DOMAIN_WHOIS;
		$last_lookup_time_field = $pdo->getLastLookupTimeField($luwhat);
		$mysqldate = $pdo->MySQLDateOffset(date("Y-m-d"),0,0,$maxdays);
		$domaintable = $pdo->getDomainTableName();
		$catconntable = $pdo->getCategoryIDToDomainIDTableName();
		$sql = "SELECT DISTINCT a.sid,a.domain FROM $domaintable a, $catconntable b WHERE (a.registrar_expiry <= ? OR a.registry_expiry <= ?) AND ((a.sid = b.sid AND b.CategoryID = ?) OR $catid=1)";	
		try
		{
			$q = $pdo->db_connect_handle->prepare($sql);
			$q->execute(array($mysqldate, $mysqldate, $catid));
			$rows = $q->fetchAll(PDO::FETCH_ASSOC);
			if($rows !== false && is_array($rows))
			{
				foreach($rows as $row)
				{
					$sid = $row['sid'];
					$domain = $row['domain'];
					if($pdo->addToLookupQueue($sid, $luwhat, $ri) !== false)
					{
						$added++;
					}
				}
			}
		}
		catch (PDOException $e) 
		{
			$pdo->setError($e->getMessage());
		}
	}
	return $added;
}

###############################################################################

function killLookupQueue($catid, $lookuptypes)
{
	$deleted = 0;
	$pdo = init_db();
	if($pdo !== false)
	{
		$luqtable = $pdo->getLookupQueueTableName();
		if(strtolower(trim($lookuptypes)) == "all")
		{
			return $pdo->deleteFromTable($luqtable);
		}
		else
		{
			$catconntable = $pdo->getCategoryIDToDomainIDTableName();
			$lutcodes = explode(",", $lookuptypes);
			foreach($lutcodes as $lucode)
			{
				$luwhat = getLookupTypeFromCode($lucode);
				$clause = "type=?";
				if($catid != 1)
				{
					$clause .= " AND (sid in SELECT DISTINCT sid FROM $catconntable WHERE CategoryID=?)";
					$dcount = $pdo->deleteFromTable($luqtable, $clause, array($luwhat, $catid));
					if($dcount !== false)
						$deleted += $dcount;
				}
				else
				{
					$dcount = $pdo->deleteFromTable($luqtable, $clause, array($luwhat));
					if($dcount !== false)
						$deleted += $dcount;
				}
			}
		}
	}
	return $deleted;
}

###############################################################################
?>
