<?php
///////////////////////////////////////////////////////////////////////////////////////////////////
#  Author:   Anil Kumar
#  Version:	 1.0
#  Date:     10-July-2013
#  Purpose:  
#  Requires: PHP 5.2+, MySQL 4.1.3+
#  Last Modified: Sep 13, 2013
#  Usage
#
#  		require_once("pdo/cppdo.php");
#		require_once("base.php");
#		require_once("auth/basic.auth.php");
#
///////////////////////////////////////////////////////////////////////////////////////////////////
#
#  Acknowledgements
#  ----------------
#  Some parts of this code were inspired by UserCake v2 
#  http://usercake.com
#  UserCake created by: Adam Davis
#  UserCake v2.0 designed by: Jonathan Cassels
#  MIT Licence
#
///////////////////////////////////////////////////////////////////////////////////////////////////



///////////////////////////////////////////////////////////////////////////////////////////////////

class cpUserManager extends cpPDO
{
	public $user_groups_table 		= null;
	public $user_table   			= null;
	public $auth_log_table   		= null;
	public $category_access_table 	= null;
	public $group_access_table      = null;
	public $table_prefix            = null;
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function __construct($table_prefix="")
	{ 
		$this->table_prefix				= trim(strtolower($table_prefix));
		$this->user_groups_table 		= trim(strtolower($table_prefix . "auth_usergroups"));
		$this->user_table   			= trim(strtolower($table_prefix . "auth_users"));
		$this->category_access_table 	= trim(strtolower($table_prefix . "auth_domain_categories"));
		$this->group_access_table 		= trim(strtolower($table_prefix . "auth_group_access"));
		$this->auth_log_table   		= trim(strtolower($table_prefix . "auth_logs"));
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function getColumns($tablename)
	{
		$columns = array();
		if(strtolower($tablename) == $this->user_groups_table)
		{
			$columns[] = "id";
			$columns[] = "int(11)";
			$columns[] = "NOT NULL AUTO_INCREMENT";
			
			$columns[] = "name";
			$columns[] = "varchar(255)";
			$columns[] = "NOT NULL";
		}
		else if(strtolower($tablename) == $this->auth_log_table)
		{
			$columns[] = "ip";
			$columns[] = "varchar(24)";
			$columns[] = "NOT NULL";
			
			$columns[] = "login_last_failed_at";
			$columns[] = "TIMESTAMP";
			$columns[] = "DEFAULT '0000-00-00 00:00:00'";
			
			$columns[] = "login_last_succeeded_at";
			$columns[] = "TIMESTAMP";
			$columns[] = "DEFAULT '0000-00-00 00:00:00'";
			
			$columns[] = "login_fail_count";
			$columns[] = "int(11)";
			$columns[] = "DEFAULT 0";
		}
		else if(strtolower($tablename) == $this->user_table)
		{
			$columns[] = "id";
			$columns[] = "int(11)";
			$columns[] = "NOT NULL AUTO_INCREMENT";
			
			$columns[] = "name";
			$columns[] = "varchar(255)";
			$columns[] = "NOT NULL DEFAULT ''";
			
			$columns[] = "display_name";
			$columns[] = "varchar(50)";
			$columns[] = "NOT NULL DEFAULT ''";
				
			$columns[] = "password";
			$columns[] = "varchar(225)";
			$columns[] = "NOT NULL";
			
			$columns[] = "email";
			$columns[] = "varchar(255)";
			$columns[] = "DEFAULT NULL";
			
			$columns[] = "active";
			$columns[] = "tinyint(1)";
			$columns[] = "NOT NULL DEFAULT 0";
			
			$columns[] = "readwrite";
			$columns[] = "tinyint(1)";
			$columns[] = "NOT NULL DEFAULT 1";
			
			$columns[] = "full_name";
			$columns[] = "varchar(225)";
			$columns[] = "DEFAULT NULL";
			
			$columns[] = "last_sign_in_stamp";
			$columns[] = "TIMESTAMP";
			$columns[] = "DEFAULT '0000-00-00 00:00:00'";
			
			$columns[] = "added_on";
			$columns[] = "TIMESTAMP";
			$columns[] = "DEFAULT CURRENT_TIMESTAMP";
		}
		else if(strtolower($tablename) == $this->category_access_table)
		{
			$columns[] = "id";
			$columns[] = "bigint(20)";
			$columns[] = "NOT NULL AUTO_INCREMENT";
			
			$columns[] = "user_id";
			$columns[] = "bigint(20)";
			$columns[] = "NOT NULL";
			
			$columns[] = "cat_id";			# domain category id.
			$columns[] = "bigint(20)";
			$columns[] = "NOT NULL";
		}
		else if(strtolower($tablename) == $this->group_access_table)
		{
			$columns[] = "id";
			$columns[] = "bigint(20)";
			$columns[] = "NOT NULL AUTO_INCREMENT";
			
			$columns[] = "user_id";
			$columns[] = "int(11)";
			$columns[] = "NOT NULL";
			
			$columns[] = "group_id";			
			$columns[] = "int(11)";
			$columns[] = "NOT NULL";
		}
		return $columns;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function getDefaultKeys($tablename)
	{
		if(strtolower($tablename) == $this->user_groups_table)
			return "PRIMARY KEY (id)";
		else if(strtolower($tablename) == $this->user_table)
			return "PRIMARY KEY (id)";
		else if(strtolower($tablename) == $this->category_access_table)
			return "PRIMARY KEY (id)";
		else if(strtolower($tablename) == $this->group_access_table)
			return "PRIMARY KEY (id)";
		else if(strtolower($tablename) == $this->auth_log_table)
			return "PRIMARY KEY (ip)";
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function getDefaultAutoIncrement($tablename)
	{
		if(strtolower($tablename) == $this->user_groups_table)
			return 3;
		else if(strtolower($tablename) == $this->category_access_table)
			return 2;
		else
			return 1;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function initTables()
	{
		$this->initTable($this->user_groups_table);
		$this->initTable($this->user_table);
		$this->initTable($this->category_access_table);
		$this->initTable($this->group_access_table);
		$this->initTable($this->auth_log_table);
		
		# add the default data (admin and 'newuser')
		$this->addUserGroup("Admin", 1);
		$this->addUserGroup("New User", 2);
		
		# add any missing or new fields
		$this->repairTable($this->user_groups_table);
		$this->repairTable($this->user_table);
		$this->repairTable($this->category_access_table);
		$this->repairTable($this->group_access_table);
		$this->repairTable($this->auth_log_table);
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function userNameExists($name)
	{
		$dberror = false;
		$status = $this->isEntryPresent($this->user_table, "name", $name, $dberror);
		if($dberror == false)
			return $status;
			
		# actually a db error occurred. But for the purpose of this function, we will ignore and 
		# return false. The db error message will be in the log.
		
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function displayNameExists($display_name)
	{
		$dberror = false;
		$status = $this->isEntryPresent($this->user_table, "display_name", $display_name, $dberror);
		if($dberror == false)
			return $status;
			
		# actually a db error occurred. But for the purpose of this function, we will ignore and 
		# return false. The db error message will be in the log.
		
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function emailExists($email)
	{
		$dberror = false;
		$status = $this->isEntryPresent($this->user_table, "email", $email, $dberror);
		if($dberror == false)
			return $status;
			
		# actually a db error occurred. But for the purpose of this function, we will ignore and 
		# return false. The db error message will be in the log.
		
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function addAuthLog($login_status=true)
	{
		$ip = $_SERVER['REMOTE_ADDR'];
		
		# Check if the IP is in the DB. If not, add it.
		$dberror = false;
		$status = $this->isEntryPresent($this->auth_log_table, "ip", $ip, $dberror);
		if($dberror == false && $status == false)
		{
			try
			{
				$stmt = $this->db_connect_handle->prepare("INSERT INTO " . $this->auth_log_table . " (ip) VALUES(:ip)");
				$stmt->execute(array(':ip' => $ip));
			}
			catch (PDOException $e) 
			{
				$this->setError($e->getMessage());
				return false;
			}
		}
		
		try
		{
			$failcount = $this->getSingleEntry($this->auth_log_table, "ip", $ip, "login_fail_count");
			if($failcount !== false)
			{
				if(!$login_status)
				{
					$failcount++;
					$stmt = $this->db_connect_handle->prepare("UPDATE " . $this->auth_log_table . " SET login_last_failed_at=?,login_fail_count=? WHERE ip=?");
				}
				else
				{
					$failcount = 0;
					$stmt = $this->db_connect_handle->prepare("UPDATE " . $this->auth_log_table . " SET login_last_succeeded_at=?,login_fail_count=? WHERE ip=?");
				}
				$stmt->execute(array(date("Y-m-d H:i:s"), $failcount, $ip));
				return true;
			}
			else
				return false;
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function addUserGroup($name, $gid=0)
	{
		$dberror = false;
		$status = $this->isEntryPresent($this->user_groups_table, "name", $name, $dberror);
		if($dberror == false && $status == false)
		{
			try
			{
				if($gid != 0)
				{
					$stmt = $this->db_connect_handle->prepare("INSERT INTO " . $this->user_groups_table . " (id,name) VALUES(:id,:name)");
					$stmt->execute(
					array(':id' => $gid, ':name' => $name));
				}
				else
				{
					$stmt = $this->db_connect_handle->prepare("INSERT INTO " . $this->user_groups_table . " (name) VALUES(:name)");
					$stmt->execute(
					array(':name' => $name));
				}
				return true;
			}
			catch (PDOException $e) 
			{
				$this->setError($e->getMessage());
				return false;
			}
		}
		else if($dberror == false)
			$this->setError("The group '" . $name . "' already exists.");
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function addUserToGroup($uid, $groupname)
	{
		$gid = $this->getSingleEntry($this->user_groups_table, "name", $groupname, "id");
		if($gid !== false)
		{
			try
			{
				$stmt = $this->db_connect_handle->prepare("SELECT * FROM " . $this->group_access_table . " WHERE user_id=? AND group_id=?");
				$stmt->execute(array($uid, $gid));
				$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
				if(count($rows) == 0)
				{
					$stmt = $this->db_connect_handle->prepare("INSERT INTO " . $this->group_access_table . " (user_id,group_id) VALUES(:user_id,:group_id)");
					$stmt->execute(
					array(':user_id' => $uid, ':group_id' => $gid));
					return true;
				}
				else
				{
					$msg = "The user id '" . $uid . "' is already a member of '" . $groupname . "'.";
					$this->setError($msg);
					return false;
				}
			}
			catch (PDOException $e) 
			{
				$this->setError($e->getMessage());
				return false;
			}
		}
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function removeUserFromGroup($uid, $groupname)
	{
		$gid = $this->getSingleEntry($this->user_groups_table, "name", $groupname, "id");
		if($gid !== false)
		{
			try
			{
				$stmt = $this->db_connect_handle->prepare("SELECT * FROM " . $this->group_access_table . " WHERE user_id=? AND group_id=?");
				$stmt->execute(array($uid, $gid));
				$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
				if(count($rows) == 1)
				{
					$stmt = $this->db_connect_handle->prepare("DELETE FROM " . $this->group_access_table . " WHERE user_id=? AND group_id=?");
					$stmt->execute(array($uid, $gid));
					return true;
				}
				else
				{
					$msg = "The user id '" . $uid . "' is not a member of '" . $groupname . "'.";
					$this->setError($msg);
					return false;
				}
			}
			catch (PDOException $e) 
			{
				$this->setError($e->getMessage());
				return false;
			}
		}
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function createUser($username, $displayname, $password, $group = "New User", $active=true, $readwrite=true, $email="")
	{
		$status = true;
		if($username == "" || $password == "")
		{		
			$this->setError("Empty user name or password.");
			$status = false;
		}
		if($this->userNameExists($username))
		{		
			$this->setError("User '" . $username . "' already exists.");
			$status = false;
		}
		if($this->displayNameExists($displayname))
		{
			$this->setError("Display name '" . $displayname . "' already exists.");
			$status = false;
		}
		if($email != "")
		{
			if(!filter_var($email, FILTER_VALIDATE_EMAIL)) 
			{
				$this->setError("Email address '" . $email . "' is not valid.");
				$status = false;
			}
			else if($this->emailExists($email))
			{
				$this->setError("Email address '" . $email . "' already exists.");
				$status = false;
			}
		}
		
		if($status == true)
		{
			$securepass = generateHash($password);
			try
			{
				$stmt = $this->db_connect_handle->prepare("INSERT INTO " . $this->user_table . " (name,display_name,password,added_on,email,active,readwrite) VALUES(:name,:display_name,:password,:added_on,:email,:active,:readwrite)");
				$stmt->execute(
				array(
				':name' => $username, ':display_name' => $displayname, ':password' => $securepass, ':added_on' => date("Y-m-d H:i:s"), ':email' => $email, ':active'=>$active, ':readwrite'=>$readwrite
				));
				$uid = $this->db_connect_handle->lastInsertId('id');
				
				if($group != "")
					$this->addUserToGroup($uid, $group);
				
				return true;
			}
			catch (PDOException $e) 
			{
				$this->setError($e->getMessage());
				return false;
			}
		}
		return $status;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function setCategoriesForUser($userid, $permissions)
	{
		if(is_array($permissions))
		{
			try
			{
				# first delete exiting.
				$sql = "DELETE FROM " . $this->category_access_table . " WHERE user_id = ?";
				$stmt = $this->db_connect_handle->prepare($sql);
				$stmt->execute(array($userid));
				
				$categoryTable = trim(strtolower($this->table_prefix . "catlist"));
				foreach($permissions as $cat)
				{
					$catid = $this->getSingleEntry($categoryTable, "CategoryName", $cat, "CategoryID");
					$stmt = $this->db_connect_handle->prepare("INSERT INTO " . $this->category_access_table . " (user_id,cat_id) VALUES(:userid,:catid)");
					$stmt->execute(array(':userid' => $userid, ':catid' => $catid));
				}
				return true;
			}
			catch (PDOException $e) 
			{
				$this->setError($e->getMessage());
				return false;
			}
		}
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function isValidLogin($username, $password)
	{
		try
		{
			$stmt = $this->db_connect_handle->prepare("SELECT * FROM " . $this->user_table . " WHERE name=? AND active=1");
			$stmt->execute(array($username));
			$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
			if(count($rows) == 1)
			{
				$userid = $rows[0]['id'];
				$storedHash = $rows[0]['password'];
				$userpass = generateHash($password, $storedHash);
				if($userpass == $storedHash)
					return true;
				else
					return false;
			}
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
		}
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function checkLogin($username, $password)
	{
		try
		{
			$stmt = $this->db_connect_handle->prepare("SELECT * FROM " . $this->user_table . " WHERE name=? AND active=1");
			$stmt->execute(array($username));
			$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
			if(count($rows) == 1)
			{
				$userid = $rows[0]['id'];
				$storedHash = $rows[0]['password'];
				$userpass = generateHash($password, $storedHash);
				
				if($userpass == $storedHash)
				{
					$this->addAuthLog(true);
					
					$userinfo['fullname'] = $rows[0]['full_name'];
					$userinfo['displayname'] = $rows[0]['display_name'];
					$userinfo['name'] = $rows[0]['name'];
					$userinfo['readwrite'] = $rows[0]['readwrite'];
					$userinfo['admin'] = false;
					$userinfo['lastsignin'] = $rows[0]['last_sign_in_stamp'];
					
					$sql = "SELECT user_id, group_id FROM " . $this->group_access_table . " WHERE user_id = ? AND group_id = 1";
					$stmt = $this->db_connect_handle->prepare($sql);
					$stmt->execute(array($userid));
					$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
					if(count($rows) == 1)
						$userinfo['admin'] = true;
					
					// Find all access rights for the user.
					$categoryTable = trim(strtolower($this->table_prefix . "catlist"));
					
					$sql = "SELECT user_id, cat_id FROM " .  $this->category_access_table . " WHERE user_id = ?";
					$stmt = $this->db_connect_handle->prepare($sql);
					$stmt->execute(array($userid));
					$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
					
					$permissions = array();
					foreach($rows as $row)
					{
						$entry['name'] = $this->getSingleEntry($categoryTable, "CategoryID", $row['cat_id'], "CategoryName");
						$entry['id'] = $row['cat_id'];
						$permissions[]  = $entry;
					}			
					
					$stmt = $this->db_connect_handle->prepare("UPDATE " . $this->user_table . " SET last_sign_in_stamp=? WHERE id=?");
					$stmt->execute(array(date("Y-m-d H:i:s"), $userid));
					
					do_login($userinfo, $permissions);
					return true;
				}
				else
				{
					$this->addAuthLog(false);
					return false;
				}
			}
			else
			{
				$this->addAuthLog(false);
				return false;
			}
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function getUserInfo($userid)
	{
		try
		{
			$stmt = $this->db_connect_handle->prepare("SELECT * FROM " . $this->user_table . " WHERE id=?");
			$stmt->execute(array($userid));
			$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
			if(count($rows) == 1)
			{
				$userinfo['fullname'] = $rows[0]['full_name'];
				$userinfo['displayname'] = $rows[0]['display_name'];
				$userinfo['name'] = $rows[0]['name'];
				$userinfo['readwrite'] = $rows[0]['readwrite'];
				$userinfo['admin'] = false;
				$userinfo['lastsignin'] = $rows[0]['last_sign_in_stamp'];
				
				$sql = "SELECT user_id, group_id FROM " . $this->group_access_table . " WHERE user_id = ? AND group_id = 1";
				$stmt = $this->db_connect_handle->prepare($sql);
				$stmt->execute(array($userid));
				$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
				if(count($rows) == 1)
					$userinfo['admin'] = true;
				
				// Find all access rights for the user.
				$categoryTable = trim(strtolower($this->table_prefix . "catlist"));
				
				$sql = "SELECT user_id, cat_id FROM " .  $this->category_access_table . " WHERE user_id = ?";
				$stmt = $this->db_connect_handle->prepare($sql);
				$stmt->execute(array($userid));
				$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
				
				$permissions = array();
				foreach($rows as $row)
				{
					$entry['name'] = $this->getSingleEntry($categoryTable, "CategoryID", $row['cat_id'], "CategoryName");
					$entry['id'] = $row['cat_id'];
					$permissions[]  = $entry;
				}			
				$userinfo['permissions'] = $permissions;
				return $userinfo;
			}
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function deleteUserGroup($name)
	{
		$groupid = $this->getSingleEntry($this->user_groups_table, "name", $name, "id");
		if($groupid !== false)
		{
			try
			{
				$sql = "DELETE FROM " . $this->user_groups_table . " WHERE " . "id =  :uid";
				$stmt = $this->db_connect_handle->prepare($sql);
				$stmt->execute(array(':uid' => $groupid));
				
				$sql = "DELETE FROM " . $this->group_access_table . " WHERE " . "group_id =  :uid";
				$stmt = $this->db_connect_handle->prepare($sql);
				$stmt->execute(array(':uid' => $groupid));
				
				return true;
			}
			catch (PDOException $e) 
			{
				$this->setError($e->getMessage());
				return false;
			}
		}
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function deleteUser($name)
	{
		$userid = $this->getSingleEntry($this->user_table, "name", $name, "id");
		if($userid !== false)
			return deleteUserID($userid);
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function deleteUserID($userid)
	{
		try
		{
			$sql = "DELETE FROM " . $this->user_table . " WHERE " . "id =  :uid";
			$stmt = $this->db_connect_handle->prepare($sql);
			$stmt->execute(array(':uid' => $userid));
			
			$sql = "DELETE FROM " . $this->category_access_table . " WHERE " . "user_id =  :uid";
			$stmt = $this->db_connect_handle->prepare($sql);
			$stmt->execute(array(':uid' => $userid));
			
			$sql = "DELETE FROM " . $this->group_access_table . " WHERE " . "user_id =  :uid";
			$stmt = $this->db_connect_handle->prepare($sql);
			$stmt->execute(array(':uid' => $userid));
			return true;
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function resetUserPassword($name, $password)
	{
		$userid = $this->getSingleEntry($this->user_table, "name", $name, "id");
		if($userid !== false)
			return $this->resetUserIDPassword($userid, $password);
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function resetUserIDPassword($userid, $password)
	{
		$securepass = generateHash($password);
		try
		{
			$stmt = $this->db_connect_handle->prepare("UPDATE " . $this->user_table . " SET password=? WHERE id=?");
			$stmt->execute(array($securepass, $userid));
			return true;
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function changeUserIDName($userid, $name)
	{
		try
		{
			$stmt = $this->db_connect_handle->prepare("UPDATE " . $this->user_table . " SET name=? WHERE id=?");
			$stmt->execute(array($name, $userid));
			return true;
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function changeUserIDDisplayName($userid, $name)
	{
		try
		{
			$stmt = $this->db_connect_handle->prepare("UPDATE " . $this->user_table . " SET display_name=? WHERE id=?");
			$stmt->execute(array($name, $userid));
			return true;
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function changeUserIDStatus($userid, $status)
	{
		try
		{
			$stmt = $this->db_connect_handle->prepare("UPDATE " . $this->user_table . " SET active=? WHERE id=?");
			$stmt->execute(array($status, $userid));
			return true;
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	public function changeUserIDReadWrite($userid, $status)
	{
		try
		{
			$stmt = $this->db_connect_handle->prepare("UPDATE " . $this->user_table . " SET readwrite=? WHERE id=?");
			$stmt->execute(array($status, $userid));
			return true;
		}
		catch (PDOException $e) 
		{
			$this->setError($e->getMessage());
			return false;
		}
		return false;
	}
	
}

///////////////////////////////////////////////////////////////////////////////////////////////////
?>
